<?php
/** 
 * Copyright 2015 Local Thanks(sm), Inc.
 *
 * File: User.php
 * Localthanks User Class
 * @author Michael W Hayden
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 */
namespace Localthanks;

use Localthanks\Merchant;
use Localthanks\Category;
use Localthanks\Utils;

class Search extends Utils
{
	// List item results and map
	public $success, $notify, $count, $results, $map;
	
	// Change of GEO location
	public $locality;
	
	// GEO Params for independent searches.
	// ll defaults to user location.
	// lat and lng are the center of a map search.
	public $ll, $lat, $lng;

	// Google
	public $ga = parent::GOOGLE_ANALYTICS_ID;
	public $gc = parent::GOOGLE_API_KEY_CLIENT;
	
	// User search details
	public $user_agent, $request_url, $request;
	
	// Var to send locations 
	// to Map Constructor
	private $locations = array();
	
    /*
     *  Build it in the construct
     */
    function __construct( $ip=null )
    {
		// Device detection
		$this->user_agent  = $_SERVER['HTTP_USER_AGENT'];

		// Request details
		$this->request_url = static::environment().$_SERVER['REQUEST_URI'];
		$this->request     = $_REQUEST;
    	
		// Establish user location 	
    	parent::userLocation( $ip );
		$this->ll          = $_REQUEST['ll']; 
		
		// Then check if locality is set.
		// Locality updates lat and lng since
		// they represent the center of a search.
		$this->locality    = static::getLocality();
 		
		// Latitude & Longitude
		$this->lat         = static::getLat();
		$this->lng         = static::getLng();
				
		// Search parameters
    	$this->mode        = static::getMode();
		$this->limit       = static::getLimit();
		$this->offset      = static::getOffset();
		$this->order       = static::getSort();
		$this->radius      = static::getRadius();
		$this->cat         = static::getCat();
		$this->term        = static::getTerm();

		// Init
			
		// Create result object and find search results.
		static::searchResults();
    }
    
    

    /*
     * Get search results
     *
     * Priority
     * Identify Local Thanks Merchant ID's
     * 
     * Get Merchants or Promotions 
     * Notes: overide category and terms 
     * Uses Keywords / Terms Category Search
     */
    private function searchResults()
    {
		$param[':limit']  = $this->limit;
		$param[':offset'] = $this->offset;
		$param[':sinlat'] = $this->lat;
		$param[':coslat'] = $this->lat;
		$param[':coslng'] = $this->lng;
			
		$ltm_id = static::search( 'ltm_id' );
		$ltp_id = static::search( 'ltp_id' );
 /*
		// Select SQL statement
		// Get Promotions by ltp_id's
		if( $ltp_id ) : 
			$sql = static::selectPromotionsByID( $ltp_id );
			$param[':ltp_id'] = $ltp_id;
		// Get Merchants by ltm_id's
		elseif( $ltm_id ) :
			switch( $this->mode ) 
			{
				// Merchants
				case 1 : $sql = static::selectMerchantsByID( $ltm_id ); break;
					
				// Promotions
				case 0 :
				default: $sql = static::selectMerchantPromotionsByID( $ltm_id ); break;
			}
			$param[':ltm_id'] = $ltm_id;
*/
		// Look by lat, lng
		if( $this->lat && $this->lng ) :
			switch( $this->mode ) 
			{
				// Merchants
				case 1 : $sql = static::selectMerchantsByLocation( $ltm_id ); break;
					
				// Promotions
				case 0 :
				default: $sql = static::selectMerchantPromotionsByLocation( $ltm_id ); break;
			}
			// bounding box area
			$area = static::bounds( $this->lat, $this->lng, $this->radius );
			$param[':minlat'] = $area->minlat;
			$param[':maxlat'] = $area->maxlat;
			$param[':minlng'] = $area->minlng;
			$param[':maxlng'] = $area->maxlng;
			
		endif;
 
 		// terms? 
		if(strpos($sql, ':term1'))
			$param[':term1'] = $this->term;
		if(strpos($sql, ':term2')) 
			$param[':term2'] = $this->term;
		if(strpos($sql, ':term3'))
			$param[':term3'] = $this->term;

		// Execute the query and format
		if( $sql ) {
			//echo var_dump( $sql );
			$dbo = static::connect();
			$qry = $dbo->prepare( $sql );
			// Loop check $prm binded values
			if( isset( $param ) ) {
				foreach( $param as $k => $v ) {
					$qry->bindValue($k, $v);
					$sql =str_replace($k, $v, $sql); // for debug
				}
			}
			//echo var_dump( $sql );
			
			$qry->execute();
			$data = $qry->fetchAll( \PDO::FETCH_OBJ );
			$data = static::processResults( $data );
			
			// debug sql
			//$this->sql = $sql;
		} // end $sql
			  
		$this->results = ( !empty($data) ) ? $data : array();
		

		// Additional API results
		// 
		// Loads remote API's, and
		// tests for results. If none
		// are set, then it moves 
		// along gracefully.
        if( !isset($_REQUEST['ltm_id']) || !isset($_REQUEST['ltp_id']) ) :
		switch( $this->mode ) {
		
			case '0' :
				// Sqoot API
				// looking for promotions 
				// via affiliates
				$sqoot = new Sqoot( $userLocation );
				if( $sqoot->results ) {
					foreach( $sqoot->results as $v ) {
						array_push( $this->results, $v );
						array_push( $this->locations, $v->geometry );
					}
				}
			break;
				
			case '1' :
				// Google API
				// looking for merchants
				// via Google places
				$google = new GooglePlaces();
				if( $google->results ) {
					foreach( $google->results as $v ) {
						array_push( $this->results, $v );
						array_push( $this->locations, $v->geometry );
					}
				}
			break;
		} // end API's
        endif;
        
		// Clean up results.
		$this->results = $this->results;
		
		// Check status
		$this->map     = new Map( $this->locations, $this->lat, $this->lng, $this->radius );
		$this->count   = count( $this->results );	
		$this->success = ( $this->count > 0 ) ? true : false;
		$this->notify  = ( $this->count > 0 ) ? static::RESULTS_YES : static::RESULTS_NO;
    }

    
	/*
	 * Search for Local Thanks Merchant ID's
	 *
	 * Merchants 
	 * if @param $_REQUEST['ltm_id']
	 * Search by ltm_id's
	 *
	 * Promotions
	 * else if @param $_REQUEST['ltp_id']
	 * Search by ltp_id's
	 *
	 * Notes: overide category and terms 
	 *
	 * Keywords / Terms
	 * else @param $_REQUEST['term']
	 * Get ltm_id's by keyword or (term)s
	 *
	 * Category Search
	 * Get all ltm_id's by (cat)egories
	 */
	const SQL_MERCH_KEYWORDS = '
		SELECT DISTINCT lt_merchants.ltm_id
		FROM lt_merchants
		INNER JOIN tbl_m2keyword ON lt_merchants.ltb_id=tbl_m2keyword.ltb_id
		INNER JOIN tbl_keywords ON tbl_m2keyword.key_id=tbl_keywords.key_id
		:inner 
		WHERE lt_merchants.ltm_company_name LIKE :name 
		OR lt_merchants.ltm_city LIKE :city 
		OR lt_merchants.ltm_zipcode LIKE :zip 
		OR tbl_keywords.key_word LIKE :word
		:and
		AND lt_merchants.ltm_active!=0 
		LIMIT :limit OFFSET :offset';
	const SQL_MERCH_INNER = 'INNER JOIN lt_promos ON lt_merchants.ltb_id=lt_promos.ltb_id ';
	const SQL_MERCH_AND   = 'OR lt_promos.ltp_headline LIKE :and ';
	
	const SQL_MERCH_CAT = '
		SELECT DISTINCT lt_merchants.ltm_id
		FROM lt_merchants
		INNER JOIN lt_m2cat ON lt_merchants.ltb_id = lt_m2cat.ltb_id
		INNER JOIN tbl_category ON lt_m2cat.cat_id = tbl_category.cat_id
		WHERE tbl_category.cat_id=:id
		AND tbl_category.cat_active !=0
		LIMIT :limit OFFSET :offset';
	
	public function search( $key )
	{
		if( isset( $_REQUEST[$key] ) && !empty( $_REQUEST[$key] ) ) {
			$str = static::sanitize( $_REQUEST[$key] );
		}
		else {
			$arr = array();
			$dbo = static::connect();
			
			// Get ltm_id's by Term
			if( !empty( $this->term ) )
			{	
				$trm = explode(' ', urldecode( $this->term ) );
				$sql = str_replace( ':inner',(($this->mode==0)? static::SQL_MERCH_INNER:''), static::SQL_MERCH_KEYWORDS );
				$sql = str_replace( ':and',  (($this->mode==0)? static::SQL_MERCH_AND:''), $sql );
								
				foreach($trm as $term) {
					$qry = $dbo->prepare( $sql );
					if($this->mode==0) 
					$qry->bindValue(':and',    "%{$term}%",   \PDO::PARAM_STR );
					$qry->bindValue(':name',   "%{$term}%",   \PDO::PARAM_STR );
					$qry->bindValue(':city',   "{$term}%",    \PDO::PARAM_STR );
					$qry->bindValue(':word',   "%{$term}%",   \PDO::PARAM_STR );
					$qry->bindValue(':zip',    "{$term}%",    \PDO::PARAM_STR );
					$qry->bindValue(':limit',  $this->limit,  \PDO::PARAM_INT );
					$qry->bindValue(':offset', $this->offset, \PDO::PARAM_INT );
					$qry->execute();
					$qry = $qry->fetchAll( \PDO::FETCH_OBJ );
					
					if( !empty ( $qry ) ) 
						foreach( $qry as $k => $v ) 
							array_push( $arr,  $v->ltm_id );
				}
			}
			
			// Get ltm_id's by Category
			if(  !empty( $this->cat ) )
			{
				$cat = explode(',', urldecode( $this->cat ) );
				foreach( $cat as $id ) {
					$qry = $dbo->prepare( static::SQL_MERCH_CAT );
					$qry->bindValue(':id',     $id);
					$qry->bindValue(':limit',  $this->limit,  \PDO::PARAM_INT );
					$qry->bindValue(':offset', $this->offset, \PDO::PARAM_INT );
					$qry->execute();
					$qry = $qry->fetchAll( \PDO::FETCH_OBJ );
					
					if( !empty($qry) ) 
						foreach( $qry as $k => $v ) 
							array_push($arr, $v->ltm_id);
				}
			}
			
			$str = implode(',', array_unique( $arr ));
		}
		
		return ( empty( $str ) ) ? '': $str;
	}

	
    
    /*
     * Creates map objects and
     * checks for merchant promotion lists.
     */
	const SQL_PROMO_LIST = "
        SELECT :fields 
        FROM lt_promos 
        WHERE lt_promos.ltb_id=:ltb_id 
        AND NOW() 
            BETWEEN lt_promos.ltp_startdate AND lt_promos.ltp_enddate 
        AND lt_promos.ltp_active!=0";
		
    public function processResults( $data )
    {
    	$dbo = static::connect();
    	$sql = str_replace( ':fields', static::promoFields(), static::SQL_PROMO_LIST );
    	$i   = 0;
    	
		foreach( $data as $k => $v ) 
		{					
			// Additional Merchant Promotions
			$qry = $dbo->prepare( $sql );
			$qry->bindValue( ':ltb_id', $v->ltb_id );
			$qry->execute();
			$data[$k]->promotions   = $qry->fetchAll( \PDO::FETCH_OBJ );
			
			if(!empty($data[$k]->promotions)) {
				foreach($data[$k]->promotions as $p) {
					$p->image = static::dynamicImgDir($p->image);
					$p->thumbnail = static::dynamicImgDir($p->thumbnail);
				}
			}
					
			// They want us baby!
			$data[$k]->provider     = static::PROJECT_NAME; 
			
			// Phone numbers
			$data[$k]->phone_number = static::phoneFormat( $v->phone_number );
			if( strpos($data[$k]->phone_number, ' or ') !== false ) {
				$number = explode(' or ', $data[$k]->phone_number);
				$data[$k]->phone_number  = $number[0];
				$data[$k]->phone_number2 = $number[1];
			}
			// search and replace merchant directory
			$data[$k]->ltb_thumbnail = static::dynamicImgDir($data[$k]->ltb_thumbnail);
			$data[$k]->ltb_image     = static::dynamicImgDir($data[$k]->ltb_image);
			$data[$k]->thumbnail     = static::dynamicImgDir($data[$k]->thumbnail);
			$data[$k]->image         = static::dynamicImgDir($data[$k]->image);
 	 		// QR Code data.
 	 		$qr = array( 
 	 			'data' => $v->ltm_url,
 	 			'size' => static::CODE_QR_SIZE
 	 		);
			$data[$k]->code_qr = static::CODE_QR.http_build_query( $qr );

			$data[$k]->distance = $data[$k]->distance.' '.static::distanceUnit();
			
			// Add ids, counts, and image paths
			$v->promo_count = count( $data[$k]->promotions );
 			$v->id          = ( $v->ltp_id ) ? implode('_',array($v->id, $v->ltp_id)): $v->id;
 			$v->thumbnail   = ($this->mode==0) ? 
 				static::dynamicImgDir($v->thumbnail): 
 				static::dynamicImgDir($v->ltb_thumbnail);
	
			// Create a map object location
			$data[$k]->geometry = new MapObject( $v->id, $v->name, $v->ltm_lat, $v->ltm_long, $v->ltm_url, $v->thumbnail, $v->promo_count );
 
			// Send location to locations array,
			// and set id to limit one map object.
			if( $i != $v->ltm_id ) 
				array_push( $this->locations, $data[$k]->geometry );
			$i = $v->ltm_id;
			
			// clean up empty vals for 
			// less data transfer.
			foreach( $v as $o => $n ) if ( empty( $n ) ) unset( $v->$o );
			unset( $v->ltm_lat, $v->ltm_long, $v->promo_count );
		}
		
		return $data;
    }

}



 
 
 
 
 
interface Google
{
	const ANALYTICS_ID   ='UA-69987455-1';
	const API_KEY_SERVER ='AIzaSyBpkTgztTfHpEZ-8KNZ6XgMuv0M9G1ASlI';
	const API_KEY_CLIENT ='AIzaSyCtEgdiaxHP7pF9NJp060I84QzyvS2q_iI';
	const URL_PHOTOS     ='https://maps.google.com/maps/api/place/photo?';
	const URL_PLACES     ='https://maps.google.com/maps/api/place/nearbysearch/json?';	
	const URL_GEOCODE    ='https://maps.google.com/maps/api/geocode/json?';
	const URL_DIRECTIONS ='https://maps.google.com/?'; // ?q=param
}
 

interface SearchLocation
{	
	public function searchLat(); // area of focus lat
	public function searchLng(); // area of focus lng
	public function deviceLat(); // device sent geo position lat
	public function deviceLng(); // device sent geo position lng
	public function ipAddrLat(); // ip from network cookie lat
	public function ipAddrLng(); // ip from network cookie lng
	public function post_code(); // post code, fall back for all if set.
	public function location();  // location info
}

interface SearchModifier
{
	public function category(); 
	public function term();    
	
	// Data Modifiers
	public function limit();   
	public function offset(); 
	public function orderBy();   
	
	// Distance 
	public function radius();
	public function distUnit();
}

interface SearchInterface 
extends Key, SearchLocation, SearchModifier 
{	
	// Default settings
	public function uid();
	public function mid();
	public function pid();
}	

class Search2 implements SearchInterface
{
	// Default settings
	const PROVIDER='Local Thanks';
	const LOGO='/cdn/images/logo2.png';
	
	const DEFAULT_RADIUS=25;
	const DEFAULT_LIMIT=25;

	// Declare Vars
	public    $success=0, $errors=array(), $notify=array(),
			  $category, $term, $terms, $post_code, $location,
			  $limit, $offset, $uid, $mid, $pid, 
			  $searchLat, $searchLng, $radius, $distUnit,
			  $count=0, $results, $env; 
			  
	protected $deviceLat, $deviceLng, 
			  $ipAddrLat, $ipAddrLng, 
			  $ip, $locality;
	
	// Database
	private   $db;
	
	// Common regular expressions
	const RGX_ALPHANUM='/[^0-9a-zA-Z]/';
	const RGX_ALNUMSPC='/[^0-9a-zA-Z ]/';
	const RGX_POSTCODE='/\b\d{5}(-\d{4})?\b/';
	
    // Construct
    function __construct($ip=null, $env=null)  
    {
    	// Set environment
		$this->ip        = $ip;
		$this->env       = $env;
		$this->logo      = $env.static::LOGO;
		$this->uri       = $_SERVER['REQUEST_URI'];
		// Set vars
    	$this->radius    = self::radius();
    	$this->distUnit  = self::distUnit();
		$this->limit     = self::limit();
		$this->offset    = self::offset();
		// Set location
		Utils::IPLocation($ip);
    	$this->searchLat = self::searchLat();
		$this->searchLng = self::searchLng();
		$this->deviceLat = self::deviceLat();
		$this->deviceLng = self::deviceLng();
		$this->ipAddrLat = self::ipAddrLat();
		$this->ipAddrLng = self::ipAddrLng();	
		// Search specifics
		$this->uid       = self::uid();       // User ID
		$this->mid       = self::mid();       // Merchant ID
		$this->pid       = self::pid();       // Product ID
		$this->category  = self::category();  // Category
		$this->term      = self::term();      // Terms		
 		$this->post_code = self::post_code(); // Post code
 		$this->location  = self::location();  // Location
    }
 

	// Init
	// Set all variables available in the request
	// @class Utils, self
 	public function init()   
 	{
    	$this->db =new Database;
    	
 		// Merchant db interface
		$sql = Merchant::SQL;
		
		// Merchants
		$mids= (empty($this->mid)) ? array(0):
		array_filter(explode(',', $this->mid));
		
  		// Result Object
 		$this->results = new \stdClass();
 		
 		// Result title regex vars
		$rgx =array(
			'/[\s\t\n\r]/' => '_',
			'/[^A-Za-z0-9\-\_]/' => ''
		);	
		$rkey=array_keys($rgx);
		$rval=array_values($rgx);

		// Custom ordering.   see function
		$order="\n".
		'ORDER BY LENGTH(product.count) DESC, '.
		 self::orderBy() . "\n".
		"LIMIT :limit OFFSET :offset \n";
		
		// Create Params array
		$params= array();
		$params[':env'] = $this->env;
		$params[':uid'] = $this->uid;
		$params[':lat'] = $this->searchLat;
		$params[':lng'] = $this->searchLng;
		
		// Add device location
     	if(empty($params[':lat']) or empty($params[':lng'])) {
     	$params[':lat'] = $this->deviceLat;
    	$params[':lng'] = $this->deviceLng;  }
    	
    	// IP address region  location fallback
     	if(empty($params[':lat']) or empty($params[':lng'])) {
     	$params[':lat'] = $this->ipAddrLat;
    	$params[':lng'] = $this->ipAddrLng;  }
 
    	// Search modifiers
    	$params[':radius']   = $this->radius;
    	$params[':distUnit'] = $this->distUnit;
		$params[':limit']    = $this->limit;
		$params[':offset']   = $this->offset;
		
		// Bounding box
		$bounds =Utils::bounds(
			$params[':lat'],
			$params[':lng'],
			$params[':radius']
        );
		$params[':minlat'] = $bounds->minlat;
		$params[':maxlat'] = $bounds->maxlat;
		$params[':minlng'] = $bounds->minlng;
		$params[':maxlng'] = $bounds->maxlng;
 
 		// Begin Search
		// Possible queries
		$this->db->beginTransaction();
		
		// Get by Merchant ID
		if(!empty($this->mid)) {
			$where = $sql.
			'WHERE merchant.ltm_id IN ('.implode(',',$mids).") \n".
			'GROUP BY merchant.ltm_id ' .$order;
			$result= self::search($where,$params);
			$count = count($result);
			if ($count>0) {
    			$this->count += $count;
    			$this->success= 1;
				$this->results->{'By Merchant'}=$result;
			}
		}
 
		$where = $sql.
		"WHERE merchant.ltm_active=1 \n";
		if(!empty($this->term)) {
		$params[':term'] = $this->term;
		$where.=
		"  AND merchant.ltm_company_name \n".
		"      LIKE CONCAT('%',:term,'%') \n";
		}
		$where.=
		"  AND merchant.ltm_lat  \n".
		"    BETWEEN ".$bounds->minlat." AND ".$bounds->maxlat." \n".
		"  AND merchant.ltm_long \n".
		"    BETWEEN ".$bounds->minlng." AND ".$bounds->maxlng." \n".
		"  AND merchant.ltm_id \n".
		'      NOT IN ('.implode(',',$mids).") \n".
		'GROUP BY merchant.ltm_id '.$order;
		
		$result = self::search($where,$params);
		$count  = count($result);
		$field  = (($count<1) ? 'No results for ':'Term ').preg_replace($rkey,$rval,$this->term);
		$this->count += count($result);
    	$this->success= 1;
		$this->results->{$field}=$result;
		foreach($result as $a => $b )
		array_push($mids, $b->mid);
 
		$this->params = $params;
		//$this->where  = $where;
		// Init Sqoot
		try {
			$SQOOT= new Sqoot2( $params, $this->category );
	        $this->results->{'More Deals'} = $SQOOT->deals();
	        $this->sqoot_uri = $SQOOT->uri;
        } 
        catch(\Exception $e) {
            $this->errors[] = $e->getMessage();
		}
		
		
/*
		// Search Merchant By Category
		foreach($this->category as $k => $v) 
		{
			$where = $sql.
			"WHERE merchant.ltm_active=1 \n".
			"  AND merchant.ltb_id IN (\n".
		    "    SELECT DISTINCT c2m.ltb_id \n".
  			"    FROM lt_m2cat AS c2m \n".
            "    WHERE c2m.cat_id= :cat_id \n".
			"    GROUP BY c2m.ltb_id \n".
			"  )\n".
			'  AND merchant.ltm_id NOT IN ('.implode(',',$mids).") \n".
			'GROUP BY merchant.ltm_id '.$order;
			$opts  = array(':cat_id' => $k );
			$result = self::search($where,array_merge($params,$opts));
			$count = count($result);
			$field = preg_replace($rkey,$rval,ucwords($v));
 
			if ($count>0) {
    			$this->count += $count;
    			$this->success= 1;
				$this->results->{$field}=$result;
				foreach($result as $a => $b )
					array_push($mids, $b->mid);
			}
		}

		// Search Merchant By Term
		if(!empty($this->post_code)) {
			$where = $sql.
			"WHERE merchant.ltm_active=1 \n".
			"  AND merchant.ltm_zipcode= :post_code \n".
			"  AND merchant.ltm_lat  \n".
			"    BETWEEN @minlat AND @maxlat \n".
			"  AND merchant.ltm_long \n".
			"    BETWEEN @minlng AND @maxlng \n".
			'  AND merchant.ltm_id NOT IN ('.implode(',',$mids).") \n".
			'GROUP BY merchant.ltm_id '.$order;
			$opts  = array(':post_code' => $this->post_code);
			$result = self::search($where,array_merge($params,$opts));
			$count = count($result);
			if ($count>0) {
    			$this->count += $count;
    			$this->success= 1;
				$this->results->{'Near post code '.$this->post_code}=$result;
				foreach($result as $a => $b )
				array_push($mids, $b->mid);
			}
		}
 
		// Search By Geo Location
		if(!empty($this->deviceLat) or !empty($this->deviceLng)) {
			$where = $sql.
			"WHERE merchant.ltm_active=1 \n".
			"  AND merchant.ltm_lat  \n".
			"    BETWEEN @minlat AND @maxlat \n".
			"  AND merchant.ltm_long \n".
			"    BETWEEN @minlng AND @maxlng \n".
			'  AND merchant.ltm_id NOT IN ('.implode(',',$mids).") \n".
			'GROUP BY merchant.ltm_id '.$order;
			$params[':lat']=$this->deviceLat;
			$params[':lng']=$this->deviceLng;
			$result = self::search($where,$params);
			$count = count($result);
			if ($count>0) {
    			$this->count += $count;
    			$this->success= 1;
				$this->results->{'Near your location'}=$result;
				foreach($result as $a => $b )
					array_push($mids, $b->mid);
			}
		}
			
		// Search Merchant By Location
		if(!empty($this->location)) {
			$where = $sql.
			"WHERE merchant.ltm_active=1 \n".
			"  AND merchant.ltm_lat  \n".
			"    BETWEEN @minlat AND @maxlat \n".
			"  AND merchant.ltm_long \n".
			"    BETWEEN @minlng AND @maxlng \n".
			'  AND merchant.ltm_id NOT IN ('.implode(',',$mids).") \n".
			'GROUP BY merchant.ltm_id '.$order;
			$result = self::search($where,$params);
			$count = count($result);
			$field = (($count<1) ? 'No results near ':'Near ').
			ucwords(preg_replace($rkey, $rval, $this->location));
			$this->count += $count;
    		$this->success= 1;
			$this->results->{$field}=$result;
			foreach($result as $a => $b )
			array_push($mids, $b->mid);
		}
*/
		// run it all together
  		$this->db->endTransaction();
    }
    
    // Datbase search
    // @var $sql, @lat, @lng
    // @option $key = object key string, no spaces.
    private function search($sql,$opt) {
    	// Prepare the query
  		$this->db->query($sql);
  		// Set default param bindings
  		foreach($opt as $k=>$v)
  		$this->db->bind($k, $v);
        ///echo '<pre>';
        //echo var_dump($sql);
        //echo $this->db->debugDumpParams();
        //echo '</pre>';
        // Run it
  		if ($this->db->execute()) {
  			$psql  = Product::SQL;
  			$result=$this->db->result();
  	    	foreach($result as $k => $v) {
  	    		if ($list=$result[$k]->product_list) {
					$result[$k]->products=
					self::search(
						$psql."WHERE product.ltp_id IN ($list)", 
						array(
							':env'=>$this->env,
							':uid'=>$this->uid,
							':ide'=>$result[$k]->id
						)
					);
    			}
    		}
    	} else {
    		$this->errors[]="$name fetch error.";
    	}
    	return $result;
    }
    
	// Data sorting 
	//
	// Add available SQL fields here.
	// ASC and DESC can be appended.
	// @request order
	public  $orderBy;
	private $orderOpts=
	array(
		0=>'distanceASC', 
		1=>'distanceDESC',
		2=>'merchant_nameDESC',
		3=>'merchant_nameASC'
	);
	// A word about PDO binding to ORDER BY
	// You can't. Use the preset order options
	// in the array and append to the SQL.
	public function orderBy() 
	{
		$this->orderBy = $_REQUEST[static::ORDER_BY];
		$key = 0;
		$rgx = array('/(ASC|DESC)/'=>' $0');
    	if (array_key_exists($this->orderBy, $this->orderOpts))
    		$key = $this->orderBy;
    	elseif (in_array($this->orderBy, $this->orderOpts, true))
			$key = array_search($this->orderBy, $this->orderOpts);
		return preg_replace(array_keys($rgx), array_values($rgx), $this->orderOpts[$key]);
	}
	
	// Available Post Code
	// Uses zip code regex to find and santize string.
	// @request post_code, term
	// @interface Google
	public function post_code() { 
		$zip = $_REQUEST[static::POSTCODE];
		$zip = (empty($zip)) ? $this->location: $zip;
    	$zip = preg_match(static::RGX_POSTCODE, $zip, $code);	
		return $code[0];
	}
	// Search by location
	// @api Google
	// @request location, post_code
	public function location() {
		$location = $_REQUEST[static::LOCATION];
		if(empty($location) and ($this->deviceLat and $this->deviceLng))
			$location = implode(',',array($this->deviceLat,$this->deviceLng));
		elseif(!preg_match(static::RGX_POSTCODE, $location)
		and preg_match(static::RGX_POSTCODE, $this->post_code))
			$location = $this->post_code;

		if ($location) {
			$uri = Google::URL_GEOCODE.http_build_query(
			array('address' => Utils::vicinityURLFormat($location)));
			// Check Google by
			// Supressing PHP warning and looking for
			// failure before preceeding.
			if(!$data=@file_get_contents($uri)) {
      			$e=error_get_last();
      			$this->errors[]='Google HTTP request failed. '.$e['message'];
			} else {
      			$data =json_decode($data);
      			$location = $data->results[0]->formatted_address;
      			// Add to object
      			$this->searchLat=$data->results[0]->geometry->location->lat;
				$this->searchLng=$data->results[0]->geometry->location->lng;
				$this->locality =$data;
			}
			// If post code not set, but Location is set 
			if (empty($this->post_code)) {
    			$zip=preg_match(static::RGX_POSTCODE,$location,$code);
				$this->post_code=$code[0];	
			}
		}
		return $location;
	}

	// Terms
	// Check term for a post_code.
	// @class Utils
	public function term() { 
		$this->terms = explode(' ',
		preg_replace(static::RGX_ALNUMSPC,'',trim($_REQUEST[static::TERM])));
		foreach($this->terms as $k => $v) {
    		if ($zip=preg_match(static::RGX_POSTCODE,$v,$code)) {
    			$_REQUEST[static::POSTCODE]=
    			$_REQUEST[static::LOCATION]=$code[0];
    			unset($this->terms[$k]);
    		}
    	}
		return implode(' ',$this->terms);
	}
	

	// Categories
	// @request cat
	public function category() {
		$category = array();
 		try {
			$this->db = (!is_object($this->db)) ? new Database: $this->db;
			$sql ='SELECT cat_slug FROM tbl_category WHERE cat_id=:id LIMIT 1';
			foreach(explode(',', $_REQUEST[static::CAT]) as $id) {
				$this->db->query($sql);
 				$this->db->bind(':id', $id);
   				if ($this->db->execute()) {
   					$res = $this->db->result();
   					$category[$id] = $res[0]->cat_slug;
				}
  			}
  			$category = array_filter($category);
        } catch ( \PDOException $e ) {
            $this->errors[] = $e->getMessage();
		}
		return $category;
	}
	

	// Helpers
 
	// User ID
	public function uid() { 
		return 14; //intval($_REQUEST[static::UID]);  
	}
	// Merchant ID
	public function mid() { 
		return trim($_REQUEST[static::MID]);     
	}
	// Product ID
	public function pid() { 
		return trim($_REQUEST[static::PRODUCT]);   
	}
	// Location Management
	public function searchLat() {
		return floatval($_REQUEST[static::SEARCH_LAT]);
	}
	public function searchLng() { 
		return floatval($_REQUEST[static::SEARCH_LNG]);
	}
	public function deviceLat() { 
		return floatval($_REQUEST[static::DEVICE_LAT]);
	}
	public function deviceLng() { 
		return floatval($_REQUEST[static::DEVICE_LNG]);
	}
	public function ipAddrLat() { 
		return floatval($_REQUEST[static::IPADDR_LAT]);
	}
	public function ipAddrLng() { 
		return floatval($_REQUEST[static::IPADDR_LNG]);
	}
	
	// Search Modifier
	// Defaults for data request size.
	// @request limit, offset
	public function limit() { 
		$limit=intval($_REQUEST[static::LIMIT]);
		return (empty($limit)) ? static::DEFAULT_LIMIT: $limit; 
	}
	public function offset() { 
		$offset=intval($_REQUEST[static::OFFSET]);
		return (empty($offset)) ? 0: $offset;
	}
	
	// Search Radius
	public function radius() { 
		$radius=intval($_REQUEST[static::RADIUS]);
		return (empty($radius)) ? static::DEFAULT_RADIUS: $radius;
	}
	
	// Distance Unit
	public function distUnit() {
        $distUnit = $_REQUEST[static::DIST_UNIT];
		if (in_array($distUnit,array('kilometer','kilo','km','k')))
			 return 'km';
		elseif (in_array($distUnit,array('nautical','nmi','nm','n')))
			 return 'nm';
		else return 'mi';
	}

} // end class Search2
 

class Sqoot2 
{	
	private $lat, $lng, $distUnit;
	
	const API_KEY ='dx2afn';
	const IMG_SIZE='&geometry=600x';
	const DEALS   ='http://api.sqoot.com/v2/deals';
	const COUPONS ='http://api.sqoot.com/v2/coupons';
	const MERCHANT='http://api.sqoot.com/v2/merchants/'; //:id?api_key=xxx
 
		
    function __construct($params, $category=null) 
    {
		// Lat & lng for distance calculation
    	$this->lat = $params[':lat'];
    	$this->lng = $params[':lng'];
    	// Distance Unit
    	$this->distUnit = $params[':distUnit'];
    	// Build query
		$this->query = http_build_query(
			array_filter(array(
				'api_key'        =>  static::API_KEY,
			    'category_slugs' =>  implode(',',$category),
				'location'       =>  implode(',',array($this->lat,$this->lng)),
				'query'          =>  $params[':term'],
				'radius'         =>  $params[':radius'],
				'per_page'       =>  $params[':limit'],
				'page'           =>(($params[':offset']/$params[':limit']) + 1)
			))
		);
    	
    	// Create the URI's 
    	//$this->coupon_uri = implode('?',array(self::COUPONS, $this->query));
	}    	

	// Process sqoot deals request
	public function deals() {
		$this->uri=implode('?',array(self::DEALS, $this->query));
		// Check Sqoot by
		// Supressing PHP warning and look for failure before preceeding.
		if(!$data=@file_get_contents($this->uri)) {
      		$e=error_get_last();
      		$this->errors[]='Sqoot HTTP request failed. '.$e['message'];
		} 
		else {
			$deals =  
			$check = array();
      		$sqoot = json_decode($data);
      		
      		foreach($sqoot->deals as $v) {
      			$key = preg_replace('/[^A-Za-z]/','',implode('',array(
      				$v->deal->merchant->name,
      				$v->deal->title
      			)));
				
      			if(in_array($key, $check)) {
      				foreach($deals as $k) {
      					if ($k->nonce == $key)
      					array_push($k->products, self::productFormat($v->deal));
      					unset($k->nonce);
      					$k->product_count = count($k->products);
      				}
      			} else {
      				array_push($deals, self::merchantFormat($v->deal, $key));
      				array_push($check, $key);
      			}
      		}
		}
		return $deals;
	} // end function
 	
	// Format the output 
	private function merchantFormat($data, $key) 
	{
		$obj = new \stdClass();
		// Merchant info
		$obj->id = "sqt-".$data->merchant->id;
		$obj->sqoot = 1;
		$obj->nonce = $key;
		$obj->mid = $data->merchant->id;
		$obj->merchant_name = ucwords(strtolower($data->merchant->name));
		$obj->address = $data->merchant->address;
		$obj->city = ucwords(strtolower($data->merchant->locality));
		$obj->state = $data->merchant->region;
		$obj->post_code = $data->merchant->postal_code;
		$obj->country = strtoupper($data->merchant->country_code);
		$obj->vicinity = Utils::vicinityFormat(
			$data->merchant->address, 
			$obj->city, 
			$obj->state, 
			$obj->post_code
		);
		$obj->merchant_lat = floatval($data->merchant->latitude);
		$obj->merchant_lng = floatval($data->merchant->longitude);
		$obj->get_directions = Google::URL_DIRECTIONS.'q='.(($obj->vicinity) ?
			Utils::vicinityURLFormat($obj->vicinity): implode(',',array($obj->merchant_lat,$obj->merchant_lng)));
		$obj->merchant_website = filter_var($data->merchant->url,FILTER_VALIDATE_URL);
		$obj->merchant_phone_number = Utils::phoneFormat($data->merchant->phone_number);
		$obj->distance = ($data->online) ? 'Online Only': implode(' ',array(
			Utils::distance(
				$this->lat,
				$this->lng,
				$obj->merchant_lat,
				$obj->merchant_lng
			),  $this->distUnit
		)); 
		$obj->locality = implode(' ',array_filter(
			array($obj->city,$obj->state,$obj->post_code)));
		// Product info
		$obj->products = array();
		array_push($obj->products, self::productFormat($data));
		$obj->product_count = 1;
		// Details
		$obj->commission = trim($data->commission);
		$obj->provider = trim($data->provider_name);
		$obj->category = trim($data->category_name);

		return (object) array_filter((array) $obj);
	}
	
	// Product formatting
	public function productFormat($data) 
	{
		$obj = new \stdClass();
		setlocale(LC_MONETARY,'en_US.UTF-8');
		// Build the object
		$obj->id = $data->id;
		$obj->title = trim(strtoupper(strip_tags($data->title)));
		$obj->type = ($data->type) ? $data->type: 0;
		$obj->start_date = Utils::dateFormat($data->created_at,'M d Y');
		$obj->expires = Utils::dateFormat($data->expires_at,'M d Y');
		$obj->dayexp = Utils::dateDiff($obj->end_date);
		$obj->url = filter_var($data->url, FILTER_VALIDATE_URL);
		$obj->image = $data->image_url.'&geometry=600x';
		$obj->merchant_lat = floatval($data->merchant->latitude);
		$obj->merchant_lng = floatval($data->merchant->longitude);
		$obj->vicinity = Utils::vicinityFormat(
			$data->merchant->address,
			$data->merchant->locality,
			$data->merchant->region,
			$data->post_code
		); 
		$obj->get_directions = 
			Google::URL_DIRECTIONS.'q='.
			Utils::vicinityURLFormat($obj->vicinity);
		$obj->description = preg_replace("/[\n\n]+/","\n", 
			strip_tags(trim($data->description)) //,'<ul><ol><li><p>')
		);
		//$obj->fine_print = strip_tags($data->fine_print,'<ul><ol><li><p>');
		$obj->number_sold = $data->number_sold;
		$obj->price = money_format('%.2n',$data->price);
		$obj->value = money_format('%.2n',$data->value);
		$obj->discount_amount = money_format('%.2n',$data->discount_amount);
		$obj->discount_percent = ($data->discount_percentage * 100). '%';
 
		return (object) array_filter((array) $obj);
	}
} // end class Sqoot2